<?php

/**
 * Copyright © 2003-2024 The Galette Team
 *
 * This file is part of Galette (https://galette.eu).
 *
 * Galette is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Galette is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Galette. If not, see <http://www.gnu.org/licenses/>.
 */

declare(strict_types=1);

namespace Galette\IO;

use Galette\Core\Preferences;
use Analog\Analog;
use Galette\Core\Login;
use Galette\Entity\Group;

/**
 * Groups list PDF
 *
 * @author Johan Cwiklinski <johan@x-tnd.be>
 */

class PdfGroups extends Pdf
{
    public const SHEET_FONT = self::FONT_SIZE - 2;

    private string $doc_title;

    /**
     * Main constructor, set creator and author
     *
     * @param Preferences $prefs Preferences
     */
    public function __construct(Preferences $prefs)
    {
        $this->filename = __('groups_list') . '.pdf';
        parent::__construct($prefs);
        // Enable Auto Page breaks
        $this->SetAutoPageBreak(true, $this->footer_height + 10);
    }

    /**
     * Draws PDF page Header
     *
     * @return void
     */
    public function Header(): void // phpcs:ignore PSR1.Methods.CamelCapsMethodName
    {
        $this->Cell(
            0,
            10,
            _T("Members by groups"),
            0,
            0,
            'C',
            false,
            '',
            0,
            false,
            'M',
            'M'
        );
    }

    /**
     * Initialize PDF
     *
     * @return void
     */
    public function init(): void
    {
        // Set document information
        $this->doc_title = _T("Members by groups");
        $this->SetTitle($this->doc_title);
        $this->SetSubject(_T("Generated by Galette"));

        // Set colors
        $this->SetTextColor(0, 0, 0);

        // Set margins
        $this->setMargins(10, 25);
        $this->setHeaderMargin(10);

        // Set font
        $this->SetFont(self::FONT, '', self::SHEET_FONT);

        //enable pagination
        $this->showPagination();

        parent::init();
    }

    /**
     * Draw file
     *
     * @param array<Group> $groups Groups list
     * @param Login        $login  Login instance
     *
     * @return void
     */
    public function draw(array $groups, Login $login): void
    {
        $this->PageHeader($this->doc_title);

        $first = true;
        foreach ($groups as $group) {
            $id = $group->getId();
            if (!$login->isGroupManager($id)) {
                Analog::log(
                    'Trying to export group ' . $id . ' as PDF without appropriate permissions',
                    Analog::INFO
                );
                continue;
            }
            // Header
            if ($first === false) {
                $this->ln(5);
            }
            $this->SetFont('', 'B', self::SHEET_FONT + 1);
            $this->Cell(190, 4, $group->getName(), 0, 1, 'C');
            $this->SetFont('', '', self::SHEET_FONT);

            $managers_list = $group->getManagers();
            $managers = array();
            foreach ($managers_list as $m) {
                $managers[] = $m->sfullname;
            }
            if (count($managers) > 0) {
                $this->Cell(
                    190,
                    4,
                    _T("Managers:") . ' ' . implode(', ', $managers),
                    0,
                    1,
                    ($this->i18n->isRTL() ? 'L' : 'R')
                );
            }
            $this->ln(3);

            $this->SetFont('', 'B');
            $this->SetFillColor(255, 255, 255);
            $this->Cell(80, 7, _T("Name"), 1, 0, 'C', true);
            $this->Cell(50, 7, _T("Email"), 1, 0, 'C', true);
            $this->Cell(30, 7, _T("Phone"), 1, 0, 'C', true);
            $this->Cell(30, 7, _T("GSM"), 1, 1, 'C', true);

            $this->SetFont('', '');

            $members = $group->getMembers();

            foreach ($members as $m) {
                $align = ($this->i18n->isRTL() ? 'R' : 'L');
                $this->Cell(80, 7, $m->sname, 1, 0, $align);
                $this->Cell(50, 7, $m->email, 1, 0, $align);
                $this->Cell(30, 7, $m->phone, 1, 0, $align);
                $this->Cell(30, 7, $m->gsm, 1, 1, $align);
            }
            $this->Cell(190, 0, '', 'T');
            $first = false;
        }
    }
}
